/*
 Name:
	Create3DBox
 Version:
	2.6 (15 August 2010)
 Author:
	Charles Bordenave
 Description:
	This script creates a 3D box with user-defined layers. 
	Assign a layer to each side of the box, and the script will position and scale them to build a box of given size.
	Note that the same layer can be used for multiple sides (the script will create copies of the layer).
	For easy control, the Add Controller option parents each side layer to a 3D Null positioned at the center of the box.
 Usage:
	In After Effects CS4 or later, run the script 
	Open a comp containing at least one layer
	Click on Refresh to update the list of available layers
	Set the dimension of the box, and assign a layer to each side of the box
	Click on Create to build the box
 */


// This class represents the main class of the script
function Create3DBox()
{
	// Variable used to keep track of 'this' reference
	var create3DBox = this;

	// Create an instance of the utils class to use its functions
	var utils = new Create3DBoxUtils();

	// Script infos
	this.scriptMinSupportVersion = "9.0";
	this.scriptName = "Create3DBox.jsx";
	this.scriptVersion = "2.6";
	this.scriptTitle = "Create 3D Box";
	this.scriptCopyright = "Copyright (c) 2010 Charles Bordenave";
	this.scriptHomepage = "http://www.nabscripts.com";
	this.scriptDescription = {en:"This script creates a 3D box with user-defined layers.\\r\\rAssign a layer to each side of the box, and the script will position and scale them to build a box of given size.\\r\\rNote that the same layer can be used for multiple sides (the script will create copies of the layer).\\r\\rFor easy control, the Add Controller option parents each side layer to a 3D Null positioned at the center of the box.", 
							  fr:"Ce script crée une boîte 3D avec les calques choisis par l\\'utilisateur.\\r\\rAssigner un calque à chacune des faces de la boîte et le script va les positionner et les redimensionner pour construire une boîte d\\'une taille donnée.\\r\\rNoter que le même calque peut servir pour plusieurs faces (le script fera des copies du calque).\\r\\rPour un meilleur contrôle, l\\'option Ajouter Contrôleur parente chaque face à un Nul 3D positionné au centre de la boîte."};
	this.scriptAbout = {en:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription), 
						fr:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription)};		
	this.scriptUsage = {en:	"\u25BA In After Effects CS4 or later, run the script \\r" +
							"\u25BA Open a comp containing at least one layer \\r" +
							"\u25BA Click on Refresh to update the list of available layers \\r" +
							"\u25BA Set the dimension of the box, and assign a layer to each side of the box \\r" +
							"\u25BA Click on Create to build the box",
						fr: "\u25BA Dans After Effects CS4 ou supérieur, exécuter le script \\r" +
							"\u25BA Ouvrir une comp contenant au moins un calque \\r" +
							"\u25BA Cliquer sur Rafraîchir pour mettre à jour la liste des calques disponibles \\r" +
							"\u25BA Spécifier la taille de la boîte et assigner un calque à chaque face \\r" +
							"\u25BA Cliquer sur Créer pour construire la boîte"};
							
	// Errors
	this.requirementErr = {en:"This script requires After Effects CS4 or later.", fr:"Ce script nécessite After Effects CS4 ou supérieur."};
	this.noCompErr = {en:"A comp must be active.", fr:"Une composition doit être active."};
	this.noLayersErr = {en:"The comp must contain at least one layer.", fr:"La composition doit contenir au moins un calque."};
	this.noSelErr = {en:"You must assign a layer to each side of the box. Do not forget to Refresh layers list if you have opened a new comp or created new layers.", fr:"Vous devez assigner un calque à chacune des faces de la boîte. N'oubliez pas d'Actualiser la liste des calques si vous avez ouvert une nouvelle comp ou créé de nouveaux calques."};
	this.badLayersErr = {en:"Layer \"%s\" cannot be used as a side of the box.", fr:"Le calque \"%s\" ne peut pas être utilisé pour créer la boîte."};

	// UI strings & default settings
	this.aboutBtnName = "?";
	this.dimensionsPnlName = {en:"Dimensions", fr:"Dimensions"};
	this.widthStName = {en:"Width:", fr:"Largeur:"};
	this.heightStName = {en:"Height:", fr:"Hauteur:"};
	this.depthStName = {en:"Depth:", fr:"Profondeur:"};
	this.uniformCbName = {en:"Uniform Size", fr:"Taille uniforme"};
	this.widthEtDflt = 100;
	this.heightEtDflt = 100;
	this.depthEtDflt = 100;
	this.texturesPnlName = {en:"Textures", fr:"Textures"};
	this.refreshBtnName = {en:"Refresh", fr:"Actualiser"};
	this.listWidth = 100; // preferredSize.width of dropdownlists
	this.frontStName = {en:"Front:", fr:"Avant:"};
	this.backStName = {en:"Back:", fr:"Arrière:"};
	this.leftStName = {en:"Left:", fr:"Gauche:"};
	this.rightStName = {en:"Right:", fr:"Droite:"};
	this.bottomStName = {en:"Bottom:", fr:"Bas:"};
	this.topStName = {en:"Top:", fr:"Haut:"};
	this.addControllerCbName = {en:"Add Controller", fr:"Ajouter Contrôleur"};
	this.controllerLayerName = {en:"Controller", fr:"Contrôleur"};
	this.runBtnName = {en:"Create", fr:"Créer"};

	// Miscellaneous 
	this.nameSep = " * "; 
	this.layerNames = {en:["Front","Back","Left","Right","Bottom","Top"], fr:["Avant","Arrière","Gauche","Droite","Bas","Haut"]}; // "LayerName" ==> "Back * LayerName"

	// Creates and displays the script interface
	this.buildUI = function (thisObj)
	{
		// dockable panel or palette
		var pal = (thisObj instanceof Panel) ? thisObj : new Window("palette", this.scriptTitle, undefined, {resizeable:false});

		// resource specifications
		var res =
		"group { orientation:'column', alignment:['left','top'], alignChildren:'fill', \
			gr1: Group { \
				aboutBtn: Button { text:'" + this.aboutBtnName + "', preferredSize:[25,20], alignment:['right','center'] } \
			}, \
			gr2: Panel { orientation:'row', alignment:['fill','fill'], alignChildren:['center','center'], text:'" + utils.loc(this.dimensionsPnlName) + "', \
				widthSt: StaticText { text:'" + utils.loc(this.widthStName) + "' }, \
				widthEt: EditText { text:'" + this.widthEtDflt + "', characters:5 }, \
				heightSt: StaticText { text:'" + utils.loc(this.heightStName) + "', enabled:false }, \
				heightEt: EditText { text:'" + this.heightEtDflt + "', characters:5, enabled:false }, \
				depthSt: StaticText { text:'" + utils.loc(this.depthStName) + "', enabled:false }, \
				depthEt: EditText { text:'" + this.depthEtDflt + "', characters:5, enabled:false }, \
				uniformCb: Checkbox { text:'" + utils.loc(this.uniformCbName) + "', value:true } \
			}, \
			gr3: Panel { orientation:'row', alignment:['fill','center'], text:'" + utils.loc(this.texturesPnlName) + "', \
				gr31: Group { alignment:['center','top'], \
					refreshBtn: Button { text:'" + utils.loc(this.refreshBtnName) + "' } \
				}, \
				gr32: Group { orientation:'row', \
					gr321: Group { orientation:'column', alignChildren:['right','top'], \
						gr3211: Group { orientation:'row', \
							frontSt: StaticText { text:'" + utils.loc(this.frontStName) + "' }, \
							frontLst: DropDownList { preferredSize:['" + this.listWidth + "',20] } \
						}, \
						gr3212: Group { orientation:'row', \
							leftSt: StaticText { text:'" + utils.loc(this.leftStName) + "' }, \
							leftLst: DropDownList { preferredSize:['" + this.listWidth + "',20] } \
						}, \
						gr3213: Group { orientation:'row', \
							bottomSt: StaticText { text:'" + utils.loc(this.bottomStName) + "' }, \
							bottomLst: DropDownList { preferredSize:['" + this.listWidth + "',20] } \
						} \
					}, \
					gr322: Group { orientation:'column', alignChildren:['right','top'], \
						gr3221: Group { orientation:'row', \
							backSt: StaticText { text:'" + utils.loc(this.backStName) + "' }, \
							  backLst: DropDownList { preferredSize:['" + this.listWidth + "',20] } \
						}, \
						gr3222: Group { orientation:'row', \
							rightSt: StaticText { text:'" + utils.loc(this.rightStName) + "' }, \
							rightLst: DropDownList { preferredSize:['" + this.listWidth + "',20] } \
						}, \
						gr3223: Group { orientation:'row', \
							topSt: StaticText { text:'" + utils.loc(this.topStName) + "' }, \
							topLst: DropDownList { preferredSize:['" + this.listWidth + "',20] } \
						} \
					} \
				}, \
			}, \
			gr4: Group { orientation:'row', alignment:['fill','top'], \
				addControllerCb: Checkbox { text:'" + utils.loc(this.addControllerCbName) + "', alignment:['left','center'], value:true }, \
				runBtn: Button { text:'" + utils.loc(this.runBtnName) + "', alignment:['right','center'] } \
			} \
		}";
		pal.gr = pal.add(res);

		// event callbacks
		pal.gr.gr1.aboutBtn.onClick = function ()
		{
			utils.createAboutDlg(create3DBox.scriptAbout, create3DBox.scriptUsage);
		};

		pal.gr.gr2.widthEt.onChange = pal.gr.gr2.widthEt.onChanging = function ()
		{
			if (isNaN(this.text) || parseFloat(this.text) < 1) this.text = create3DBox.widthEtDflt;
			this.text = parseFloat(this.text);

			if (pal.gr.gr2.uniformCb.value)
			{
				pal.gr.gr2.heightEt.text = pal.gr.gr2.depthEt.text = this.text;
			}
		};

		pal.gr.gr2.uniformCb.onClick = function ()
		{
			if (this.value)
			{
				pal.gr.gr2.heightEt.text = pal.gr.gr2.depthEt.text = pal.gr.gr2.widthEt.text;
			}
			pal.gr.gr2.heightSt.enabled = pal.gr.gr2.depthSt.enabled =
			pal.gr.gr2.heightEt.enabled = pal.gr.gr2.depthEt.enabled = !this.value;
		};

		pal.gr.gr3.gr31.refreshBtn.onClick = function ()
		{
			try
			{
				var comp = app.project.activeItem;
				var err = create3DBox.noCompErr;
				if (create3DBox.checkActiveItem(comp)) throw(err);

				var lists = [
				pal.gr.gr3.gr32.gr321.gr3211.frontLst,
				pal.gr.gr3.gr32.gr321.gr3212.leftLst,
				pal.gr.gr3.gr32.gr321.gr3213.bottomLst,
				pal.gr.gr3.gr32.gr322.gr3221.backLst,
				pal.gr.gr3.gr32.gr322.gr3222.rightLst,
				pal.gr.gr3.gr32.gr322.gr3223.topLst
				];

				for (var i = 1; i <= comp.numLayers; i++)
				{
					for (var j = 0; j < lists.length; j++)
					{
						if (i == 1) lists[j].removeAll();

						lists[j].add("item", comp.layer(i).name);

						if (i == 1) lists[j].selection = 0;
					}
				}
			}
			catch(err)
			{
				utils.throwErr(err);
			}
		};

		pal.gr.gr4.runBtn.onClick = function ()
		{
			create3DBox.createBox(pal);
		};

		// show user interface
		if (pal instanceof Window)
		{
			pal.center();
			pal.show();
		}
		else
		{
			pal.layout.layout(true);
		}
	};

	// Determines whether the active item is a composition
	this.checkActiveItem = function ()
	{
		return !(app.project.activeItem instanceof CompItem);
	};

	// Creates a 3D Null and make it parent of the layers passed as arguments
	this.addController = function (layers)
	{
		var comp = layers[0].containingComp;
		var controllerLayer = comp.layers.addNull();
		controllerLayer.name = utils.loc(this.controllerLayerName);
		controllerLayer.threeDLayer = true;
		controllerLayer.enabled = false;

		var pos = [0,0,0];
		for (var i = 0; i < layers.length; i++) pos += layers[i].position.valueAtTime(comp.time, false);
		pos /= layers.length;

		var p = controllerLayer.position;
		p.numKeys ? p.setValueAtTime(comp.time, pos) : p.setValue(pos);

		for (var i = 0; i < layers.length; i++) layers[i].parent = controllerLayer;

		controllerLayer.position.setValue([comp.width/2,comp.height/2,0]);
	};

	// Scales a layer so that its size (in pixels) matches the desired size
	this.resizeToFit = function (layer, desiredSize, parFlag)
	{
		var compPar = layer.containingComp.pixelAspect;
		var layerPar = layer.source ? layer.source.pixelAspect : 1.0;
		
		var layerSize = utils.getLayerSize(layer, layer.containingComp.time);
		
		var layerW = layerSize[0] * layerPar; // real width
		var layerH = layerSize[1];		

		var targetW = desiredSize[0] * Math.pow(layerPar, Number(parFlag)+1) / layerPar; // same as   parFlag ? desiredSize[0] * layerPar : desiredSize[0];
		if (layerPar != 1.0 && layerPar != compPar && parFlag) targetW *= compPar/layerPar;
		
		var targetH = desiredSize[1];

		var sx = 100 * targetW / layerW;
		var sy = 100 * targetH / layerH;

		var s = layer.scale;
		s.numKeys ? s.setValueAtTime(layer.containingComp.time, [sx,sy]) : s.setValue([sx,sy]);
	};

	// Functional part of the script: creates a 3D box according to user settings
	this.createBox = function (pal)
	{
		try
		{
			var comp = app.project.activeItem;
			var err = this.noCompErr;
			if (this.checkActiveItem(comp)) throw(err);
			
			var err = this.noLayersErr;
			if (comp.layers.length < 1) throw(err);
			
			var err = this.noSelErr;
			try
			{
				var selfront = comp.layer(pal.gr.gr3.gr32.gr321.gr3211.frontLst.selection);
				var selback = comp.layer(pal.gr.gr3.gr32.gr322.gr3221.backLst.selection);
				var selleft = comp.layer(pal.gr.gr3.gr32.gr321.gr3212.leftLst.selection);
				var selright = comp.layer(pal.gr.gr3.gr32.gr322.gr3222.rightLst.selection);
				var selbottom = comp.layer(pal.gr.gr3.gr32.gr321.gr3213.bottomLst.selection);
				var seltop = comp.layer(pal.gr.gr3.gr32.gr322.gr3223.topLst.selection);
				if (!selfront || !selback || !selleft || !selright || !selbottom || !seltop) throw(err);
			}
			catch(e)
			{
				throw(err);
			}

			// initialization
			var front = selfront;
			var back = selback;
			var left = selleft;
			var right = selright;
			var bottom = selbottom;
			var top = seltop;

			// box settings
			var boxW = parseFloat(pal.gr.gr2.widthEt.text) / comp.pixelAspect;
			var boxH = parseFloat(pal.gr.gr2.heightEt.text);
			var boxD = parseFloat(pal.gr.gr2.depthEt.text);
			var halfW = 0.5 * boxW;
			var halfH = 0.5 * boxH;
			var halfD = 0.5 * boxD;
			var pos = [[halfW, halfH, 0], [halfW, halfH, boxD], [0, halfH, halfD], [boxW, halfH, halfD], [halfW, boxH, halfD], [halfW, 0, halfD]];
			var ori = [[0,0,0], [0,180,0], [0,90,0], [0,270,0], [90,0,0], [270,0,0]];
		
			app.beginUndoGroup(this.scriptTitle);

			// create duplicates if necessary
			if (selback == selfront) back = selback.duplicate();
			if (selleft == selfront || selleft == selback) left = selleft.duplicate();
			if (selright == selfront || selright == selback || selright == selleft) right = selright.duplicate();
			if (selbottom == selfront || selbottom == selback || selbottom == selleft || selbottom == selright) bottom = selbottom.duplicate();
			if (seltop == selfront || seltop == selback || seltop == selleft || seltop == selright || seltop == selbottom) top = seltop.duplicate();

			var layers = [front,back,left,right,bottom,top];

			// create the box
			var err = this.badLayersErr;
			try
			{
				for (var i = 0; i < layers.length; i++)
				{
					layers[i].threeDLayer = true;
					var layerPar = layers[i].source ? layers[i].source.pixelAspect : 1.0;
					var layerSize = utils.getLayerSize(layers[i], comp.time);
					var layerW = layerSize[0]; 
					var layerH = layerSize[1];
					var a = layers[i].anchorPoint;
					var p = layers[i].position;
					var r = layers[i].orientation;
					
					if (layers[i] instanceof TextLayer) // assume left justify, since cs3 does not support font stuff
					{
						// text justification
						var textDocVal = layers[i].property("ADBE Text Properties").property("ADBE Text Document").value;
						var ax = 0;
						var ay = -layerH/2;
						if (textDocVal.justification == ParagraphJustification.LEFT_JUSTIFY)
						{
							ax += layerW/2;
						}
						else if (textDocVal.justification == ParagraphJustification.RIGHT_JUSTIFY)
						{
							ax -= layerW/2;			   		
						}
						a.numKeys ? a.setValueAtTime(comp.time, [ax, ay, 0]) : a.setValue([ax, ay, 0]);					
					}
					else
						a.numKeys ? a.setValueAtTime(comp.time, 0.5 * [layerW, layerH, 0]) : a.setValue(0.5 * [layerW, layerH, 0]);
					
					if (layerPar == 1.0)
					{
						var pp = [pos[i][0]/comp.pixelAspect, pos[i][1], pos[i][2]];
						p.numKeys ? p.setValueAtTime(comp.time, pp) : p.setValue(pp);	
					}
					else
						p.numKeys ? p.setValueAtTime(comp.time, pos[i]) : p.setValue(pos[i]);
					r.numKeys ? r.setValueAtTime(comp.time, ori[i]) : r.setValue(ori[i]); 
					
					if (i == 0 || i == 1) this.resizeToFit(layers[i],[boxW,boxH], true); // front & back
					if (i == 2 || i == 3) this.resizeToFit(layers[i],[boxD,boxH], false); // left & right
					if (i == 4 || i == 5) this.resizeToFit(layers[i],[boxW,boxD], true); // bottom & top
				}
			}
			catch(e)
			{
				alert(e);
				err.en = err.en.replace('%s',layers[i].name);
				err.fr = err.fr.replace('%s',layers[i].name);
				throw(err);
			}

			// add controller if requested
			if (pal.gr.gr4.addControllerCb.value)
			{
				this.addController(layers, [halfW, -halfH, halfD]);
			}

			// rename layers					
			for (var i = 0; i < layers.length; i++)
			{
				layers[i].name = utils.loc(this.layerNames)[i] + this.nameSep + layers[i].name;
			}

			app.endUndoGroup();
		}
		catch(err)
		{
			utils.throwErr(err);
		}
	};

	// Runs the script
	this.run = function (thisObj)
	{
		if (parseFloat(app.version) < parseFloat(this.scriptMinSupportVersion))
		{
			utils.throwErr(this.requirementErr);
		}
		else
		{
			this.buildUI(thisObj);
		}
	};
}


// This class provides some utility functions
function Create3DBoxUtils()
{
	// Variable used to keep track of 'this' reference
	var utils = this;
	
	// String localization function: english and french languages are supported
	this.loc = function (str)
	{
		return app.language == Language.FRENCH ? str.fr : str.en;
	};

	// Displays a window containg a localized error message
	this.throwErr = function (err)
	{
		var wndTitle = $.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf("."));
		Window.alert("Script error:\r" + this.loc(err), wndTitle, true);
	};

	// Get the size in pixels of an AV layer (comp layer, footage layer, solid layer, text layer) at specific time
	this.getLayerSize = function (avLayer, time)
	{
		var w, h;
		if (!(avLayer instanceof TextLayer))
		{
			w = avLayer.width;
			h = avLayer.height;
		}
		else
		{
			var bb = avLayer.sourceRectAtTime(time, true);
			w = bb.width;
			h = bb.height;
		}
		return [w,h];
	};
	
	// Displays a dialog containg the script description and usage
	this.createAboutDlg = function (aboutStr, usageStr)
	{	
		eval(unescape('%09%09%76%61%72%20%64%6C%67%20%3D%20%6E%65%77%20%57%69%6E%64%6F%77%28%22%64%69%61%6C%6F%67%22%2C%20%22%41%62%6F%75%74%22%29%3B%0A%09%20%20%20%20%20%20%09%20%20%20%20%20%20%20%09%0A%09%20%20%20%20%76%61%72%20%72%65%73%20%3D%0A%09%09%22%67%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%63%6F%6C%75%6D%6E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%61%6C%69%67%6E%43%68%69%6C%64%72%65%6E%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%09%09%09%70%6E%6C%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%62%65%64%70%61%6E%65%6C%27%2C%20%5C%0A%09%09%09%09%61%62%6F%75%74%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%44%65%73%63%72%69%70%74%69%6F%6E%27%2C%20%5C%0A%09%09%09%09%09%61%62%6F%75%74%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%2C%20%5C%0A%09%09%09%09%75%73%61%67%65%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%55%73%61%67%65%27%2C%20%5C%0A%09%09%09%09%09%75%73%61%67%65%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%20%5C%0A%09%09%09%7D%2C%20%5C%0A%09%09%09%62%74%6E%73%3A%20%47%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%72%6F%77%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%62%6F%74%74%6F%6D%27%5D%2C%20%5C%0A%09%09%09%09%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%74%68%65%72%20%53%63%72%69%70%74%73%2E%2E%2E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%6C%65%66%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%2C%20%5C%0A%09%09%09%09%6F%6B%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%6B%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%09%09%09%7D%20%5C%0A%09%09%7D%22%3B%20%0A%09%09%64%6C%67%2E%67%72%20%3D%20%64%6C%67%2E%61%64%64%28%72%65%73%29%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%3B%0A%09%09%7D%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%27%2F%67%2C%20%22%27%22%29%3B%0A%09%09%7D%3B%0A%09%09%09%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%76%61%72%20%63%6D%64%20%3D%20%22%22%3B%0A%09%09%09%76%61%72%20%75%72%6C%20%3D%20%22%68%74%74%70%3A%2F%2F%61%65%73%63%72%69%70%74%73%2E%63%6F%6D%2F%63%61%74%65%67%6F%72%79%2F%73%63%72%69%70%74%73%2F%6E%61%62%2F%22%3B%0A%09%0A%09%09%09%69%66%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%0A%09%09%09%7B%0A%09%20%20%20%20%20%20%20%20%09%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%20%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%49%6E%74%65%72%6E%65%74%20%45%78%70%6C%6F%72%65%72%2F%69%65%78%70%6C%6F%72%65%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%7D%0A%09%09%09%65%6C%73%65%0A%09%09%09%09%63%6D%64%20%2B%3D%20%22%6F%70%65%6E%20%5C%22%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%20%20%20%20%20%20%20%20%20%0A%09%09%09%74%72%79%0A%09%09%09%7B%0A%09%09%09%09%73%79%73%74%65%6D%2E%63%61%6C%6C%53%79%73%74%65%6D%28%63%6D%64%29%3B%0A%09%09%09%7D%0A%09%09%09%63%61%74%63%68%28%65%29%0A%09%09%09%7B%0A%09%09%09%09%61%6C%65%72%74%28%65%29%3B%0A%09%09%09%7D%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%6B%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%0A%09%09%7B%0A%09%09%09%64%6C%67%2E%63%6C%6F%73%65%28%29%3B%20%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%63%65%6E%74%65%72%28%29%3B%0A%09%09%64%6C%67%2E%73%68%6F%77%28%29%3B'));		
	};
}


// Creates an instance of the main class and run it
new Create3DBox().run(this);
